' ----------------------------------------------------------------------------
' Type field to store a single system (simple)
' ----------------------------------------------------------------------------
Type TSystem

	Field id:Int
	Field Name:String
	Field x:Float
	Field y:Float
	Field z:Float
	Field visits:Int
	Field lastvisit:Int
	
	' create an object
	Method New(id:Int)

		'ListAddLast(systemlist, Self)

	End Method
	
End Type

Type TAnalysis

	Field cat:String
	Field Key:String
	Field val:Int

End Type

' ----------------------------------------------------------------------------
' Type field to store a single body
' ----------------------------------------------------------------------------
Type TBody

	' logical
	Field id:Int
	Field Name:String
	Field scandate:String
	Field system:String
	Field subsystem:String
	Field sysid:Int	
	
	' star+planet specific
	Field distance:Float
	Field temperature:Float
	Field radius:Float
	
	' star-specific
	Field startype:String
	Field starmass:Float
	Field magnitude:Float
	Field age:Float
	
	' planet-specific
	Field class:String
	Field planetmass:Float
	Field atmosphere:String
	Field pressure:Float
	Field gravity:Float
	Field tidallocked:String
	Field terraformable:String
	Field terraflag:Int
	Field landable:String
	Field biosigns:String
	
	Field starsystem:String
	Field starstar:String
	Field starplanet:String
	Field starmoon:String
	Field starsubmoon:String
	
	Field lastscan:Int
	
	' ----------------------------------------------------------------------------
	' Initializes the database
	' ----------------------------------------------------------------------------
	Function CreateDatabase()
	
		' --------------------------------------------------------------------
		' create new database file
		' --------------------------------------------------------------------
		LogUpdate("Creating new database", True)
		db = LoadDatabase("SQLITE", ":memory:")'title + ".db")
		sql = TDBSQLite(db)
				
		' create system table
		LogUpdate("Creating tables", True)
		q = "CREATE TABLE system (id integer primary key AUTOINCREMENT, name TEXT, x REAL, y REAL, z REAL, visits INTEGER, lastvisit INTEGER)"
		db.executeQuery(q)
		
		' create body table
		q = "CREATE TABLE body (id integer primary key AUTOINCREMENT, pid INTEGER, pstar TEXT, name TEXT, systemname TEXT, star TEXT, planet INTEGER, moon TEXT, submoon TEXT, systemid INTEGER, startype TEXT, starmass REAL, starmag REAL, starage REAL, class TEXT, distance REAL, temperature REAL, mass REAL, radius REAL, pressure REAL, gravity REAL, atmosphere TEXT, tidallocked INTEGER, terraformable INTEGER, landable INTEGER, lastscan INTEGER)"
		db.executeQuery(q)
			
		' execute SQL statements
		db.commit()
			
	End Function
	
	' ----------------------------------------------------------------------------
	' Deletes the database
	' ----------------------------------------------------------------------------
	Function DeleteDatabase()
	
		If (db) Then
		
			LogUpdate("Deleting old database", True)
			db.close()
		
			db = Null
			
		EndIf
	
	End Function

	' ----------------------------------------------------------------------------
	' scan the journal folder, evaluate and output results
	' ----------------------------------------------------------------------------
	Function RescanFolder()
	
		LogUpdate("Scan requested.")
		
		DeleteDatabase()
		CreateDatabase()
				
		' --------------------------------------------------------------------
		' read path
		' --------------------------------------------------------------------
		Local dir:Int = ReadDir(path)
		Local filename:String = ""
		Local s:TStream
		Local l:String
		
		' clear old stats
		bodycount = Null
		GCCollect()
		bodycount = New TBodyCount
		
		' clear old analysis
		For Local a:TAnalysis = EachIn analysislist
		
			a = Null
		
		Next
		ClearList(analysislist)
			
		' play scanner sound (looped)
		If blipflag Then PlaySample(9)
		
		' UNIX Timestamp: get current timestamp in Seconds sind January 1, 1970
		Local curtime:Int = GetTimeStamp()
		Local logtime:Int = 0
		Local limit:Int = timelimit
		
		' speed measurement
		Local totalms:Int = MilliSecs()
		Local timeusage:Float
		Local lps:String
						
		' read all files
		Repeat
			
			' convert filename to lowercase
			filename = Lower(NextFile(dir))
			
			' skip . and ..
			If filename = "." Or filename = ".." Then Continue
			
			' filename has log extension?
			If Left(filename, 8) = "journal." And Right(filename, 4) = ".log" Then
					
				' UNIX Timestamp: Seconds since January 1, 1970
				logtime = FileTime(path + filename)
								
				Local condition:Int = False
				
				If (curtime - logtime) <= limit Then condition = True
				If timelimit = -1 And logtime > datefrom And logtime < dateto Then condition = True
									
				' filedate is within the limit?
				If condition Then
			
					' try to open the file
					s = OpenStream(path + filename)
					
					' successful?
					If s Then
					
						' read lines until end of file
						While Not Eof(s)
						
							' read next line
							l = ReadLine(s)
							
							l = Replace(l, "~q", "'")
		
							' parse line to TBody type field
							bodycount.body = TBody.ParseLine(l, False)
							
							' count bodies
							bodycount.counter:+1
							
							' scan this line
							'bodycount.Scan()
		
							' update status bar
							If bodycount.counter Mod 1000 = 0 Then
							
								SetStatusText(Window, FormatINT(bodycount.counter) + " lines scanned.")' ; Delay 1
							
							EndIf

							'UpdateProgBar progress, (bodycount.counter Mod 1000) / 1000.0
		
						Wend
						
					EndIf
								
					' close file
					CloseStream(s)
															
					' update status bar
					SetStatusText(Window, FormatINT(bodycount.counter) + " lines scanned.")
		
				EndIf
										
			EndIf
			
		Until filename = ""
		
		' close directory
		CloseDir dir
		
		' time statistics
		timeusage = (MilliSecs() - totalms) / 1000.0			
		If bodycount.counter > 0 And timeusage > 0.0 Then lps = FormatINT(Int(bodycount.counter / timeusage))
		
		' update log
		LogUpdate("Journal Scan finished in " + FormatMath(timeusage, 3) + " seconds. " + FormatINT(bodycount.counter) + " lines scanned (" + lps + " Lines per second).")
		LogUpdate("------------------------------------------------------------")
		
		' stop scanner sound
		StopChannel channel
		
		' play blip sound
		If blipflag Then PlaySample(8)
		
		LogUpdate("Evaluating scanned objects", True)
		
		' evaluate results
		bodycount.Evaluate()
		
		LogUpdate("Filling database with values", True)
		
		Local ms:Int = MilliSecs()

		' initiate DB transaction
		db.startTransaction()
		
		' prepare inserts
		Local qu:TDatabaseQuery = TDatabaseQuery.Create(db)
		qu.prepare("INSERT INTO body values (NULL, ?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)")
				
		Local body:TBody
		Local system:TSystem
		
		' reset counter
		Local cnt:Int = 0
		
		' count total number of bodies in list
		Local c1:Int = bodylist.count()
								
		' iterate through all bodies
		For body = EachIn bodylist
			
			' iterate through all systems
			For system = EachIn systemlist
			
				' sytem name is part of the body name?
				If Instr(body.name, system.name) Then
							
					' check if system name fits
					If Left(body.name, Len(system.name) + 1) = system.name + " " Or Trim(body.name) = Trim(system.name) Then
							
						' make bodyname RegEx friendly
						Local bn:String = body.Name
		
						' assign first values
						body.starsystem = system.Name
						body.subsystem = LTrim(Replace(bn, body.starsystem, ""))
						body.sysid = system.id
							
						' RegEx pattern
						Local p1:String = "\A(?:([A-Z]{1,3}|[0-9]{0,2}))"
						Local p2:String = "?(?:\ ([0-9]{0,2}|[a-z]{0,1}))"
						Local p3:String = "?(?:\ ([a-z]{0,1}))"
						Local p4:String = "?(?:\ ([a-z]{0,1}))?\z"
						Local pattern:String = p1 + p2 + p3 + p4
							
						' perform RegEx on subsystem
						Local subsystem:String = RegFindSystem(body.subsystem, pattern)
						Local out:String[] = subsystem.split("|")
						
						' parse RegEx output string
						For Local i:Int = 0 To out.Length - 1
							
							Select i
								
								Case 0 body.starstar = Trim(out[i])
								Case 1 body.starplanet = Trim(out[i])
								Case 2 body.starmoon = Trim(out[i])
								Case 3 body.starsubmoon = Trim(out[i])
								
							End Select
								
						Next
							
						' star is a number?
						If Int(body.starstar) > 0 Then
												
							' shift planet/moon/submoon one column to the left
							body.starsubmoon = body.starmoon
							body.starmoon = body.starplanet
							body.starplanet = body.starstar
							body.starstar = Null
						
						EndIf
						
						If body.starsystem = "" Then LogUpdate("Can't find system for body: " + body.Name)
						
						' update status bar
						If cnt Mod 100 = 0 Then SetStatusText(Window, "Filling database with values: " + FormatMath(cnt * 1.0 / c1 * 100, 0) + "%")' ; Delay 1
												
						' increase counter
						cnt:+1

						UpdateProgBar progress, Int(FormatMath(cnt * 1.0 / c1 * 100, 0)) / 99.0
						
						Exit
		
					End If
				
				EndIf
								
			Next
			
			' convert flags to integer values
			Local tl:Int = Int(body.tidallocked)
			Local tf:Int = Int(body.terraflag)
			Local la:Int = Int(body.landable)
						
			If Int(body.starplanet) = 0 Then body.starplanet = Null

			qu.bindValue(0, TDBInt.Set(0))
			qu.bindValue(1, TDBString.Set("?"))
			qu.bindValue(2, TDBString.Set(body.Name))
			qu.bindValue(3, TDBString.Set(body.starsystem))
			qu.bindValue(4, TDBString.Set(body.starstar))
			qu.bindValue(5, TDBInt.Set(Int(body.starplanet)))
			qu.bindValue(6, TDBString.Set(body.starmoon))
			qu.bindValue(7, TDBString.Set(body.starsubmoon))
			qu.bindValue(8, TDBInt.Set(body.sysid))
			qu.bindValue(9, TDBString.Set(body.startype))
			qu.bindValue(10, TDBFloat.Set(body.starmass))
			qu.bindValue(11, TDBFloat.Set(body.Magnitude))
			qu.bindValue(12, TDBFloat.Set(body.age))
			qu.bindValue(13, TDBString.Set(body.Class))
			qu.bindValue(14, TDBFloat.Set(body.distance))
			qu.bindValue(15, TDBFloat.Set(body.temperature))
			qu.bindValue(16, TDBFloat.Set(body.planetmass))
			qu.bindValue(17, TDBFloat.Set(body.radius))
			qu.bindValue(18, TDBFloat.Set(body.pressure))
			qu.bindValue(19, TDBFloat.Set(body.gravity))
			qu.bindValue(20, TDBString.Set(body.atmosphere))
			qu.bindValue(21, TDBInt.Set(tl))
			qu.bindValue(22, TDBInt.Set(tf))
			qu.bindValue(23, TDBInt.Set(la))
			qu.bindValue(24, TDBInt.Set(body.lastscan))

			qu.execute()
			
			body.id = qu.lastInsertedId()
																		
		Next	
						
		cnt = 0
		Local c2:Int = systemlist.count()

		' prepare inserts
		qu.prepare("INSERT INTO system values (NULL, ?,?,?,?,?,?)")
		
		' iterate all systems
		For system = EachIn systemlist
		
			qu.bindValue(0, TDBString.Set(system.name))
			qu.bindValue(1, TDBFloat.Set(system.x))
			qu.bindValue(2, TDBFloat.Set(system.y))
			qu.bindValue(3, TDBFloat.Set(Int(system.z)))
			qu.bindValue(4, TDBInt.Set(system.visits))
			qu.bindValue(5, TDBInt.Set(system.lastvisit))
			
			qu.execute()
			
			' update status bar
			If cnt Mod 100 = 0 Then SetStatusText(Window, "Filling database with values: " + FormatMath(cnt * 1.0 / c2 * 100, 0) + "%")' ; Delay 1
			
			cnt:+1

			ListRemove(systemlist, system)
			system = Null
			
		Next
				
		LogUpdate("Database creation finished", True)
		
		' execute all SQL commands at once
		db.commit()
		
		cnt = 0
		
		Local body2:TBody
		
		qu.prepare("UPDATE body SET pid=?, pstar=?, distance=? WHERE ID=?")
			
		' iterate through all bodies
		For body = EachIn bodylist
				
			' temp var holds star system name
			Local n:String = body.starsystem
			
			' if starname uses addendum, add it
			If body.starstar Then n:+" " + Left(body.starstar, 1)
						
			For body2 = EachIn bodylist
			
				' name is identical? update body ID with current ID as PID
				If body2.name = n Then
				
					qu.bindValue(0, TDBInt.Set(body2.id))
					qu.bindValue(1, TDBString.Set(body2.startype))
					qu.bindValue(2, TDBFloat.Set(Abs(body2.distance - body.distance)))
					qu.bindValue(3, TDBInt.Set(body.id))
					qu.execute()
					Exit
					
				EndIf
								
			Next

			cnt:+1

			' update status bar
			If cnt Mod 100 = 0 Then SetStatusText(Window, "Determining body parents: " + FormatMath(cnt * 1.0 / c1 * 100, 0) + "%")' ; Delay 1
			
			UpdateProgBar progress, Int(FormatMath(cnt * 1.0 / c1 * 100, 0)) / 99.0
						
		Next
		
		For body = EachIn bodylist
			
			' cleanup type field
			ListRemove(bodylist, body)
			body = Null
		
		Next
				
		LogUpdate("Database transaction completed in " + FormatMath((MilliSecs() - ms) / 1000.0) + " seconds")
		
		' clear visits TMap
		ClearMap(systemvisits)
		
		' create Analysis Tab data
		Analize(db)
	
			
		' free unused RAM
		GCCollect()
		
		' update log
		LogUpdate("Showing results.", True)
		LogUpdate("------------------------------------------------------------")

		' play computer finish sound
		If blipflag Then AddSample(10)

	End Function
	
	' ----------------------------------------------------------------------------
	' returns the number of found bodies of a given class from the database
	' ----------------------------------------------------------------------------
	Function CountBody:Int(db:TDBConnection, cond1:String, what1:String, comp:String = "=", cond2:String = "", what2:String = "")
	
		Local countwhat:String = "mass"
		If cond1 = "startype" Then countwhat = "starmass"
		If cond1 = "name" Then countwhat = "distance"
	
		Local qu:String = "SELECT count(DISTINCT " + countwhat + ") FROM body WHERE " + cond1 + " " + comp + " '" + what1 + "'"
		If cond2 <> "" Then qu:+" AND " + cond2 + " = ~q" + what2 + "~q"
		
		Local q:TDatabaseQuery = db.executeQuery(qu)
				
		If db.hasError() Then
		
			Print db.error().toString()
			
		Else
		
			If q.nextRow() Then Return q.rowRecord().value(0).getInt()
			
		EndIf
		
	End Function
	
	' ----------------------------------------------------------------------------
	' analyze the collected data from database
	' ----------------------------------------------------------------------------
	Function Analize(db:TDBConnection)
	
		Local q:TDatabaseQuery
		Local a:TAnalysis

		' clear old stats
		'bodycount = Null
		'GCCollect()
		'bodycount = New TBodyCount
		
		bodycount.ammon = CountBody(db, "class", "Ammonia world")
		bodycount.earth = CountBody(db, "class", "Earthlike body")
		bodycount.water = CountBody(db, "class", "Water world")
		bodycount.waterterr = CountBody(db, "class", "Water world", "=", "terraformable", "1")
		bodycount.metal = CountBody(db, "class", "Metal rich body")
		bodycount.icy = CountBody(db, "class", "Icy body")
		bodycount.rocky = CountBody(db, "class", "Rocky body")
		bodycount.rockyterr = CountBody(db, "class", "Rocky body", "=", "terraformable", "1")
		bodycount.rockicy = CountBody(db, "class", "Rocky ice body")
		bodycount.hmc = CountBody(db, "class", "High metal content body")
		bodycount.hmcterr = CountBody(db, "class", "High metal content body", "=", "terraformable", "1")
		
		bodycount.gas1 = CountBody(db, "class", "%Sudarsky class I gas giant%", "LIKE")
		bodycount.gas2 = CountBody(db, "class", "%Sudarsky Class II gas giant%", "LIKE")
		bodycount.gas3 = CountBody(db, "class", "%Sudarsky Class III gas giant%", "LIKE")
		bodycount.gas4 = CountBody(db, "class", "%Sudarsky Class IV gas giant%", "LIKE")
		bodycount.gas5 = CountBody(db, "class", "%Sudarsky Class V gas giant%", "LIKE")
		
		bodycount.cluster = CountBody(db, "name", "%Belt cluster%", "LIKE")
		
		bodycount.gasammon = CountBody(db, "class", "Gas giant with ammonia based life")
		bodycount.gashelium = CountBody(db, "class", "Helium rich gas giant")
		bodycount.gaswater = CountBody(db, "class", "Gas giant with water based life")
		bodycount.watergiant = CountBody(db, "class", "Water giant")
		
		bodycount.classo = CountBody(db, "startype", "O")
		bodycount.classb = CountBody(db, "startype", "B")
		bodycount.classa = CountBody(db, "startype", "A")
		bodycount.classf = CountBody(db, "startype", "F")
		bodycount.classg = CountBody(db, "startype", "G")
		bodycount.classk = CountBody(db, "startype", "K")
		bodycount.classm = CountBody(db, "startype", "M")
		bodycount.classy = CountBody(db, "startype", "Y")
		bodycount.classt = CountBody(db, "startype", "T")
		bodycount.classl = CountBody(db, "startype", "L")
		bodycount.classtts = CountBody(db, "startype", "TTS")
		
		bodycount.classc = CountBody(db, "startype", "C")
		bodycount.classn = CountBody(db, "startype", "N")
		bodycount.classh = CountBody(db, "startype", "H")
		bodycount.classx = CountBody(db, "startype", "X")
		bodycount.classw = CountBody(db, "startype", "W")
		
		bodycount.classd:+CountBody(db, "startype", "D")
		bodycount.classd:+CountBody(db, "startype", "DA")
		bodycount.classd:+CountBody(db, "startype", "DAB")
		bodycount.classd:+CountBody(db, "startype", "DAO")
		bodycount.classd:+CountBody(db, "startype", "DAZ")
		bodycount.classd:+CountBody(db, "startype", "DAV")
		bodycount.classd:+CountBody(db, "startype", "DB")
		bodycount.classd:+CountBody(db, "startype", "DBZ")
		bodycount.classd:+CountBody(db, "startype", "DBV")
		bodycount.classd:+CountBody(db, "startype", "DO")
		bodycount.classd:+CountBody(db, "startype", "DOV")
		bodycount.classd:+CountBody(db, "startype", "DQ")
		bodycount.classd:+CountBody(db, "startype", "DC")
		bodycount.classd:+CountBody(db, "startype", "DCV")
		bodycount.classd:+CountBody(db, "startype", "DX")
				
		bodycount.giant:+CountBody(db, "startype", "M_RedGiant")
		bodycount.giant:+CountBody(db, "startype", "K_OrangeGiant")
		bodycount.supergiant:+CountBody(db, "startype", "M_RedSuperGiant")
		bodycount.supergiant:+CountBody(db, "startype", "F_WhiteSuperGiant")
		bodycount.supergiant:+CountBody(db, "startype", "A_BlueWhiteSuperGiant")
		
		bodycount.Evaluate()
		
		q = db.executeQuery("SELECT pstar, count(DISTINCT mass) as number FROM body WHERE terraformable GROUP BY pstar")
				
		While q.nextRow()
			
			Local r:TQueryRecord = q.rowRecord()
				
			If r Then
			
				Local k:String = r.getString(0)
				Local v:Int = r.getInt(1)
								
				Select k
				
					Case "O", "B", "A", "F", "G", "K", "M"
					
						a = New TAnalysis
						a.cat = "TerraStar"
						a.Key = "Star Class " + k
						a.val = v
						ListAddLast(analysislist, a)
										
				End Select
			
			EndIf

		Wend
						
		q = db.executeQuery("SELECT pstar, count(DISTINCT mass) as number FROM body WHERE class='Earthlike body' GROUP BY pstar")
				
		While q.nextRow()
			
			Local r:TQueryRecord = q.rowRecord()
				
			If r Then
			
				Local k:String = r.getString(0)
				Local v:Int = r.getInt(1)
								
				Select k
				
					Case "O", "B", "A", "F", "G", "K", "M"
					
						a = New TAnalysis
						a.cat = "EarthStar"
						a.Key = "Star Class " + k
						a.val = v
						ListAddLast(analysislist, a)
										
				End Select
			
			EndIf

		Wend
		
		q = db.executeQuery("SELECT pstar, count(DISTINCT mass) as number FROM body WHERE class='Water world' GROUP BY pstar")
				
		While q.nextRow()
			
			Local r:TQueryRecord = q.rowRecord()
				
			If r Then
				
				Local k:String = r.getString(0)
				Local v:Int = r.getInt(1)
								
				Select k
				
					Case "O", "B", "A", "F", "G", "K", "M"
					
						a = New TAnalysis
						a.cat = "WaterStar"
						a.Key = "Star Class " + k
						a.val = v
						ListAddLast(analysislist, a)
										
				End Select
			
			EndIf

		Wend
				
		q = db.executeQuery("SELECT pstar, count(*) as number FROM body GROUP BY pstar")
		
		Local dwarfs:Int = 0
		Local wolfs:Int = 0
				
		While q.nextRow()
			
			Local r:TQueryRecord = q.rowRecord()
				
			If r Then
				
				Local k:String = r.getString(0)
				Local v:Int = r.getInt(1)
				
				Select Left(k, 1)
				
					Case "O", "B", "A", "F", "G", "K", "M", "T", "N", "H", "Y", "L"
					
						a = New TAnalysis
						a.cat = "StarType"

						a.Key = "Star Class " + k
						If Instr(k, "Giant") Then a.Key = "Star Class " + Left(k, 1) + " (Giant)"
						If k = "H" Then a.Key = "Star Class Black Hole"
						If k = "N" Then a.Key = "Star Class Neutron Star"

						a.val = v
						ListAddLast(analysislist, a)
						
					Case "D"
					
						dwarfs:+v
						
					Case "W"
					
						wolfs:+v

					Case "?"
					
						a = New TAnalysis
						a.cat = "StarType"
						a.Key = "Star Class unknown"
						a.val = v
						ListAddLast(analysislist, a)

				End Select
			
			EndIf

		Wend
		
		If dwarfs > 0 Then
		
			a = New TAnalysis
			a.cat = "StarType"
			a.Key = "Star Class White Dwarf"
			a.val = dwarfs
			ListAddLast(analysislist, a)
			
		EndIf
		
		If wolfs > 0 Then
		
			a = New TAnalysis
			a.cat = "StarType"
			a.Key = "Star Class Wolf-Rayet"
			a.val = wolfs
			ListAddLast(analysislist, a)
			
		EndIf
		
	End Function
	
	
	' ----------------------------------------------------------------------------
	' parse single log line and extract relevant scan data to TBody type field
	' ----------------------------------------------------------------------------
	Function ParseLine:TBody(l:String, ignore:Int = False)
	
		Local body:TBody
		Local offset:Int = 0
					
		' get current timestamp in UNIX format
		Local curtime:Int = GetTimeStamp()
			
		' replace quotes to blitzmax-compatible style
		'l = Replace(l, "~q", "'")
		
		'Local bio:String = RegFind(l, "'FSSBodySignals':'(.*?)'")
		'LogUpdate(bio)
						
		Local ev:String = RegFind(l, "'Event':'(.*?)'")
			
		' check for system Jumps = get pure system names
		If ev = "FSDJump" or ev = "CarrierJump" Then
	
			' extract scandate using RegEx
			Local scandate:String = RegFind(l, "'timestamp':'(.*?)',")
			
			' extract star position using RegEx			
			Local starpos:String[] = RegFind(l, "'StarPos':\[(.*?)\],").split(",")
		
			' extract scantime parts from JSON scandate
			Local YY1:Int = Int(Mid(scandate, 1, 4))
			Local MM1:Int = Int(Mid(scandate, 6, 2))
			Local DD1:Int = Int(Mid(scandate, 9, 2))
			Local HH1:Int = Int(Mid(scandate, 12, 2))
			Local II1:Int = Int(Mid(scandate, 15, 2))
			Local SS1:Int = Int(Mid(scandate, 18, 2))
					
			' calculate log timestamp (hh-1 is CET-1 to match UK time)
			Local logtime1:Int = GetUnixTimestamp(yy1, mm1, dd1, hh1 - offset, ii1, ss1)
			
			Local condition:Int = False
			
			If (curtime - logtime1) <= timelimit Or ignore = True Then condition = True
			If timelimit = -1 And logtime1 > datefrom And logtime1 < dateto Then condition = True
												
			' only continue if scandate is within Scan Range time
			If condition Then
			
				' extract star system name
				Local n:String = RegFind(l, "'StarSystem':'(.*?)',")
				
				' reset unique flag
				Local unique:Int = True
				
				' cast type
				Local s:TSystem
							
				' iterate through all systems
				For Local u:TSystem = EachIn systemlist
				
					' system already exists (= we've already visited this system before!)
					If u.Name = Replace(n, "'", "''") Then
					
						' flag uniqueness
						unique = False
						
						' retrieve TSystem Object from System TMap
						s = TSystem(MapValueForKey(systemvisits, n))
						
						' increase visits counter by 1
						s.visits:+1
						s.lastvisit = logtime1
						
					EndIf
				
				Next
	
				' only create if system is yet unknown
				If unique = True Then
				
					' increase System ID (equivalent to SQL Autoincrement ID)
					sysid:+1
				
					' add new Type
					s = New TSystem
					s.id = sysid
					s.Name = n'Replace(n, "'", "''")' SQL fix
					s.visits = 1 ' only one visit yet
					
					s.x = Float(starpos[0])
					s.y = Float(starpos[1])
					s.z = Float(starpos[2])
					
					s.lastvisit = logtime1
					
					' add to System TMap
					MapInsert(systemvisits, n, s)					
					ListAddLast(systemlist, s)
					
				EndIf
	
			EndIf
					
		EndIf
		
		' check for detail scans
		If RegFind(l, "'Event':'(.*?)'") = "Scan" Then
			
			' create a new object
			body = New TBody
			ListAddLast(bodylist, body)
					
			' extract JSON scandate using RegEx
			body.scandate = RegFind(l, "'timestamp':'(.*?)',")
			
			' extract scantime parts from JSON scandate
			Local YY:Int = Int(Mid(body.scandate, 1, 4))
			Local MM:Int = Int(Mid(body.scandate, 6, 2))
			Local DD:Int = Int(Mid(body.scandate, 9, 2))
			Local HH:Int = Int(Mid(body.scandate, 12, 2))
			Local II:Int = Int(Mid(body.scandate, 15, 2))
			Local SS:Int = Int(Mid(body.scandate, 18, 2))
			
			' calculate log timestamp (hh-1 is CET-1 to match UK time)
			Local logtime:Int = GetUnixTimestamp(yy, mm, dd, hh - offset, ii, ss)

			Local condition:Int = False
			
			If (curtime - logtime) <= timelimit Or ignore = True Then condition = True
			If timelimit = -1 And logtime > datefrom And logtime < dateto Then condition = True
			
			' only perform if scan time is within scan range
			If condition Then
			
				' scan for body name using RegEx
				body.Name = RegFind(l, "'BodyName':'(.*?)',")
							
				' encapsulate ' to '' for SQL operations
				'body.Name = Replace(body.Name, "'", "''")
				body.Class = RegFind(l, "'PlanetClass':'(.*?)'")
				
				' star+planet specific
				body.distance = Float(RegFind(l, "'DistanceFromArrivalLS':(.*?),"))
				body.temperature = Float(RegFind(l, "'SurfaceTemperature':(.*?),"))
				body.radius = Float(RegFind(l, "'Radius':(.*?),"))
		
				' star-specific
				body.startype = RegFind(l, "'StarType':'(.*?)',")
				body.starmass = Float(RegFind(l, "'StellarMass':(.*?),"))
				body.magnitude = Float(RegFind(l, "'AbsoluteMagnitude':(.*?),"))
				body.age = Float(RegFind(l, "'Age_MY':(.*?),"))
		
				' planet-specific
				body.class = RegFind(l, "'PlanetClass':'(.*?)'")
				body.planetmass = Float(RegFind(l, "'MassEM':(.*?),"))
				body.atmosphere = RegFind(l, "'Atmosphere':'(.*?)'")
				body.pressure = Float(RegFind(l, "'SurfacePressure':(.*?),"))
				body.gravity = Float(RegFind(l, "'SurfaceGravity':(.*?),"))
				body.tidallocked = RegFind(l, "'TidalLock':(.*?),")
				body.terraformable = RegFind(l, "'TerraformState':'(.*?)'")
				body.landable = RegFind(l, "'Landable':(.*?),")
				
				' set numeric flags
				If body.tidallocked = "true" Then body.tidallocked = 1
				If body.terraformable = "Terraformable" Then body.terraflag = 1
				If body.landable = "true" Then body.landable = 1
				
				body.lastscan = logtime
				
			Else
			
				' delete object
				ListRemove(bodylist, body)
				body = Null
				GCCollect()
		
			EndIf

		EndIf
		
		Return body
	
	End Function
	
	
	' ----------------------------------------------------------------------------
	' reads the current CMDR's name, rank and progress once
	' ----------------------------------------------------------------------------
	Function InitCMDR()
	
		'Print currentlogfile
	
		' open this logfile
		Local s:TStream = OpenStream(path + currentlogfile)
		Local line:String
	
		' get the LAST line
		Repeat
	
			line = ReadLine(s)
				
			' and scan CMDR name, Rank and progress as we read all lines
			If (RegFind(line, "~qEvent~q:~q(.*?)~q") = "LoadGame") Then currentcmdrstring = line
			If (RegFind(line, "~qEvent~q:~q(.*?)~q") = "Rank") Then currentrankstring = line
			If (RegFind(line, "~qEvent~q:~q(.*?)~q") = "Progress") Then currentprogstring = line
	
		Until Eof(s) Or (currentcmdrstring <> "" And currentrankstring <> "" And currentprogstring <> "")
				
		CloseStream s
			
	End Function



	' ----------------------------------------------------------------------------
	' monitors journal log changes
	' ----------------------------------------------------------------------------
	Function Monitor()
	
		Local dir:Int = ReadDir(path)
		Local filename:String = ""
		Local filedate:Int = 0
		Local filedatemax:Int = 0
		Local logfile:String = ""
		
		' read all files and check for the newest file
		Repeat
		
			' get next filename
			filename = Lower(NextFile(dir))
			
			' skip . and ..
			If filename = "." Or filename = ".." Then Continue
			
			' check if it has the journal prefix and the log extension
			If Left(filename, 8) = "journal." And Right(filename, 4) = ".log" Then
			
				' get filedate
				filedate = Int(FileTime(path + filename))
													
				' get the maximum filedate (=current log!)
				If filedate > filedatemax Then
				
					filedatemax = filedate
					logfile = filename
					currentlogfile = logfile
					
				EndIf
				
			EndIf
			
		Until filename = ""
	
		CloseDir dir
		
		' no max filedate? no current logfile!
		If filedatemax = 0 Then currentlogfile = ""
	
		' update statustext
		If currentlogfile <> "" Then
	
			'statustext = activity[counter] + " Monitoring Logfile: " + currentlogfile
	
		Else
	
			statustext = "ERROR: Logfile not found. Select an appropriate path in the File menu!"
	
		EndIf
				
		If oldline = "" Then
		
			' open this logfile
			Local s:TStream = OpenStream(path + logfile)
	
			' get the LAST line
			Repeat
	
				oldline = ReadLine(s)
					
			Until Eof(s)
				
			CloseStream s
		
		EndIf
						
		' if there IS a logfile (= newest found) continue
		If logfile <> "" Then
	
			' open this logfile
			Local s:TStream = OpenStream(path + logfile)
			Local line:String
			
			' dirty hack to read only the last 10% of the log :)
			If s.size() > 1000 Then s.Seek(FileSize(path + logfile) * 0.9)
	
			' read until the LAST line, which is the only one of interes here
			Repeat
	
				line = ReadLine(s)
					
			Until Eof(s)
				
			CloseStream s
							
			' check if the last line has been changed and ONLY parse if it is different
			If line <> oldline Then
	
				oldline = line
	
				' play sound
				If blipflag Then AddSample(62) ' Blip 2
	
				LogUpdate("------------------------------------------------------------")
				LogUpdate("Logfile changed detected!")
				LogUpdate("Path: " + path)
				LogUpdate("Log: " + logfile)
				LogUpdate("Line: " + line)
				LogUpdate("------------------------------------------------------------")
	
				' replace double quotes with a single quote (to prevent SQL errors)
				line = Replace(line, "~q", "'")
		
				' check if the line contains a detail scan
				If RegFind(line, "'Event':'(.*?)'") = "Scan" Then
	
					' play sound
					If scanflag Then
					
						If sampflag = 0 Then
						
							AddSample(1) ' Drumroll
							
						Else
						
							AddSample(31) ' Scan completed
							
						EndIf
						
					EndIf
	
					' parse the line
					body = TBody.ParseLine(line, True)
	
					' evaluate stars
					If starflag Then TBody.EvaluateStars(body)
	
					' evaluate planets
					If planflag Then TBody.EvaluatePlanets(body)
	
					' play sound
					If blipflag Then AddSample(8) ' Blip 1
	
				EndIf
	
			EndIf
				
		EndIf	
			
	End Function
	
	
	
	' ----------------------------------------------------------------------------
	' Evaluation of Star scans
	' ----------------------------------------------------------------------------
	Function EvaluateStars(body:TBody)
	
		' stars
		Select body.startype
	
			' Brown Dwarfs
			' --------------------------------------------------------------------
			Case "T"
					If sampflag = 1 Then
					
						AddSample(40) ' Star Class T
						AddSample(65) ' Brown Dwarf star
						If scooflag Then AddSample(67) ' Not scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "Y"
					If sampflag = 1 Then
					
						AddSample(42) ' Star Class Y
						AddSample(65) ' Brown Dwarf star
						If scooflag Then AddSample(67) ' Not scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "L"
					If sampflag = 1 Then
					
						AddSample(37) ' Star Class L
						AddSample(65) ' Brown Dwarf star
						If scooflag Then AddSample(67) ' Not scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
			
			
			' Main Sequence Stars
			' --------------------------------------------------------------------
			Case "M"
					If sampflag = 1 Then
					
						AddSample(38) ' Star Class M
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "K"
					If sampflag = 1 Then
					
						AddSample(36) ' Star Class K
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "G"
					If sampflag = 1 Then
					
						AddSample(35) ' Star Class G
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "F"
					If sampflag = 1 Then
					
						AddSample(34) ' Star Class F
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "A"
					If sampflag = 1 Then
					
						AddSample(32) ' Star Class A
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "B"
					If sampflag = 1 Then
					
						AddSample(33) ' Star Class B
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "O"
					If sampflag = 1 Then
					
						AddSample(39) ' Star Class O
						AddSample(64) ' Main Sequence
						If scooflag Then AddSample(55) ' Scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
			
			' Special Stars
			' --------------------------------------------------------------------
			Case "X"
					If sampflag = 1 Then
					
						AddSample(22) ' Exotic Star
						If scooflag Then AddSample(67) ' Not scoopable
						AddSample(61) ' This is a very rare stellar object
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
	
			Case "W", "WN", "WNC", "WC", "WO"
			
					If sampflag = 1 Then
					
						AddSample(52) ' Wolf-Rayet Star
						If scooflag Then AddSample(67) ' Not scoopable
						AddSample(61) ' This is a very rare stellar object
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "CS", "C", "CN", "CJ", "CH", "CHd"
			
					If sampflag = 1 Then
					
						AddSample(14) ' Carbon Star
						If scooflag Then AddSample(67) ' Not scoopable
						AddSample(61) ' This is a very rare stellar object
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
	
			Case "TTS", "AeBe"
			
					If sampflag = 1 Then
					
						AddSample(41) ' Star Class T-Tauri
						AddSample(68) ' Protostar
						If scooflag Then AddSample(67) ' Not scoopable
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			' Star Remnants
			' --------------------------------------------------------------------
			Case "N"
					If sampflag = 1 Then
					
						AddSample(28) ' Neutron Star
						
						If warnflag Then
						
							AddSample(59) ' Danger
							AddSample(56) ' Extreme Radiation detected
							AddSample(60) ' Be careful
							
						EndIf
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
			
			Case "H", "SupermassiveBlackHole"
			
					If sampflag = 1 Then
					
						AddSample(13) ' Black Hole
						
						If warnflag Then
						
							AddSample(59) ' Danger
							AddSample(57) ' Extreme Gravity detected
							AddSample(60) ' Be careful
						
						EndIf
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
					
			Case "D", "DA", "DAB", "DAO", "DAZ", "DAV", "DB", "DBZ", "DBV", "DO", "DOV", "DQ", "DC", "DCV", "DX"
			
					If sampflag = 1 Then
					
						AddSample(49) ' White Dwarf
						
						If warnflag Then
						
							AddSample(59) ' Danger
							AddSample(57) ' Extreme Gravity detected
							AddSample(60) ' Be careful
							
						EndIf
						
					Else
					
						If hornflag Then AddSample(3) ' Horn
						
					EndIf
			
		End Select
	
	
		' Giant stars
		' --------------------------------------------------------------------
		If Instr(body.startype, "SuperGiant") Then
		
			If sampflag = 1 Then
					
				AddSample(44) ' Supergiant
				If scooflag Then AddSample(55) ' Scoopable
				
				If warnflag Then
		
					AddSample(69) ' Warning
					AddSample(58) ' Extreme heat detected
					AddSample(60) ' Be careful
					
				EndIf
				
			Else
			
				If hornflag Then AddSample(3) ' Horn
				
			EndIf
			
		Else If Instr(body.startype, "Giant") Then
		
			If sampflag = 1 Then
	
				AddSample(23) ' Giant
				If scooflag Then AddSample(55) ' Scoopable
				
				If warnflag Then
				
					AddSample(69) ' Warning
					AddSample(58) ' Extreme heat detected
					AddSample(60) ' Be careful
					
				EndIf
			
			Else
			
				If hornflag Then AddSample(3) ' Horn
				
			EndIf
	
		EndIf
	
	End Function
	
	
	
	' ----------------------------------------------------------------------------
	' Evaluation of Planet scans
	' ----------------------------------------------------------------------------
	Function EvaluatePlanets(body:TBody)
	
		If sampflag = 1 And Instr(body.Name, "Belt Cluster") Then AddSample(12) ' Belt Cluster
	
		' count planets
		Select body.Class
		
	
			' Very precious planets
			' --------------------------------------------------------------------
			Case "Earthlike body"
									If sampflag = 1 Then
									
										AddSample(21) ' Earthlike
										
										If precflag Then
										
											AddSample(54) ' This is a very precious planet
											'AddSample(20) ' Congratulations
											
										EndIf
										
									Else
									
										AddSample(4) ' Whoooo
										
									EndIf
										
									If cashflag Then AddSample(2)  ' Cash register
			
			Case "Water world"
									If sampflag = 1 Then
									
										AddSample(47) 'Water World
									
										If body.terraflag Then
										
											AddSample(45) ' This planet is terraformable
	
											If precflag Then
	
												AddSample(54) ' This is a very precious planet
												'AddSample(20) ' Congratulations
												
											EndIf
											
										Else
										
											If precflag Then AddSample(53) ' This is a precious planet
											
										EndIf
										
									Else
									
										AddSample(5) ' Water waves
										
										If body.terraflag Then AddSample(4) ' Whoooo
										
									EndIf
									
									If cashflag Then AddSample(2)  ' Cash register
	
			
			' Precious Planets
			' --------------------------------------------------------------------
			Case "Ammonia world"
									If sampflag = 1 Then
									
										AddSample(11) ' Ammoniaworld
										
										If precflag Then
										
											AddSample(53) ' This is a precious planet
											'AddSample(20) ' Congratulations
											
										EndIf
										
									Else
									
										AddSample(6) ' Laboratory
										
									EndIf
										
									If cashflag Then AddSample(2)  ' Cash register
			
			Case "High metal content body"
												
									If sampflag = 1 Then
									
										AddSample(25) ' High Metal Content
										
										If body.terraflag Then
																				
											AddSample(45) ' This planet is terraformable
											
											If precflag Then
																						
												AddSample(54) ' This is a very precious planet
												'AddSample(20) ' Congratulations
												
											EndIf
											
											If cashflag Then AddSample(2)  ' Cash register
											
										EndIf
										
									Else
									
										If body.terraflag Then
										
											AddSample(4) ' Whoooo
											If cashflag Then AddSample(2) ' Cash register
										
										Else
										
											If hornflag Then AddSample(3) ' Horn
											
										EndIf
										
									EndIf
									
			' Not so precious Planets
			' --------------------------------------------------------------------
			Case "Metal rich body"
									If sampflag = 1 Then
									
										AddSample(27) ' Metal Rich
										If precflag Then AddSample(53) ' This is a precious planet
										
									Else
									
										AddSample(7) ' Anvil
										
									EndIf
									
									If cashflag Then AddSample(2)  ' Cash register
	
			Case "Rocky body"
									If sampflag = 1 Then
									
										AddSample(30) ' Rocky
										
										If body.terraflag Then
										
											AddSample(45) ' This planet is terraformable
											
											If precflag Then
											
												AddSample(53) ' This is a precious planet
												
												
											EndIf
											
											AddSample(2)  ' Cash register
											
										EndIf
										
									Else
									
										If body.terraflag Then
										
											If cashflag Then AddSample(2) ' Cash Register
											
										Else
										
											If hornflag Then AddSample(3) ' Horn
											
										EndIf
										
									EndIf
									
			' Gas Giants
			' --------------------------------------------------------------------
			Case "Gas giant with ammonia based life"
			
									If sampflag = 1 Then
									
										AddSample(63) ' Sudarski Gas Giant
										AddSample(50) ' with ammonia based life
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
			
			Case "Gas giant with water based life"
			
									If sampflag = 1 Then
									
										AddSample(63) ' Sudarski Gas Giant
										AddSample(51) ' with wate based life
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
									
									EndIf
			
			Case "Helium gas giant", "Helium rich gas giant"
			
									If sampflag = 1 Then
									
										AddSample(63) ' Gas Giant
										AddSample(24) ' Helium rich
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
	
			' Sudarsky Class Gas Giants
			' --------------------------------------------------------------------
			Case "Sudarsky class I gas giant"
			
									If sampflag = 1 Then
									
										AddSample(43) ' Sudarski Gas Giant
										AddSample(15) ' Class 1 - Jovian
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
	
			Case "Sudarsky class II gas giant"
			
									If sampflag = 1 Then
									
										AddSample(43) ' Sudarski Gas Giant
										AddSample(16) ' Class 2 - Water clouds
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
									
			Case "Sudarsky class III gas giant"
			
									If sampflag = 1 Then
									
										AddSample(43) ' Sudarski Gas Giant
										AddSample(17) ' Class 3 - Cloudless
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
			
			Case "Sudarsky class IV gas giant"
			
									If sampflag = 1 Then
									
										AddSample(43) ' Sudarski Gas Giant
										AddSample(18) ' Class 4 - Alkali clouds
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
			
			Case "Sudarsky class V gas giant"
			
									If sampflag = 1 Then
									
										AddSample(43) ' Sudarski Gas Giant
										AddSample(19) ' Class 5 - Silicate clouds
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
					
			' Junk
			' --------------------------------------------------------------------
			Case "Icy body"
									If sampflag = 1 Then
	
										AddSample(26) ' Icy
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
									
			Case "Rocky ice body"
									If sampflag = 1 Then
	
										AddSample(29) ' Rocky Icy
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
									
			Case "Water giant"
									If sampflag = 1 Then
	
										AddSample(30) ' Water Giant
										
									Else
									
										If hornflag Then AddSample(3) ' Horn
										
									EndIf
		
		End Select
	
	
	End Function

End Type

' ----------------------------------------------------------------------------
' Type field to store scan results
' ----------------------------------------------------------------------------
Type TBodyCount

	Field body:TBody
	Field counter:Long

	Field classm:Long = 0
	Field classk:Long = 0
	Field classg:Long = 0
	Field classf:Long = 0
	Field classa:Long = 0
	Field classb:Long = 0
	Field classo:Long = 0
	
	Field classn:Long = 0
	Field classh:Long = 0
	Field classx:Long = 0
	Field classc:Long = 0
	Field classd:Long = 0
	Field classw:Long = 0

	Field classt:Long = 0
	Field classl:Long = 0
	Field classy:Long = 0
	Field classtts:Long = 0
	
	Field giant:Long = 0
	Field supergiant:Long = 0
	
	Field earth:Long = 0
	Field metal:Long = 0
	Field water:Long = 0
	Field ammon:Long = 0
	Field hmc:Long = 0
	Field terra:Long = 0

	Field hmcterr:Long = 0
	Field waterterr:Long = 0
	Field rockyterr:Long = 0

	Field rocky:Long = 0
	Field icy:Long = 0
	Field rockicy:Long = 0
	Field gas1:Long = 0
	Field gas2:Long = 0
	Field gas3:Long = 0
	Field gas4:Long = 0
	Field gas5:Long = 0
	Field gaswater:Long = 0
	Field gasammon:Long = 0
	Field gashelium:Long = 0
	Field watergiant:Long = 0
	Field cluster:Long = 0
	
	Field total1:Long = 0
	Field total2:Long = 0
	Field total3:Long = 0
	Field total4:Long = 0
	Field total5:Long = 0
	Field total6:Long = 0
		
	Field total:Long = 0

	' ----------------------------------------------------------------------------

	Field mclassy:Long = 0
	Field mclasst:Long = 0
	Field mclassl:Long = 0
	Field mclasstts:Long = 0
	
	Field mclassm:Long = 0
	Field mclassk:Long = 0
	Field mclassg:Long = 0
	Field mclassf:Long = 0
	Field mclassa:Long = 0
	Field mclassb:Long = 0
	Field mclasso:Long = 0
	
	Field mgiant:Long = 0
	Field msupergiant:Long = 0
	Field mclassn:Long = 0
	Field mclassd:Long = 0
	Field mclassh:Long = 0
	Field mclassc:Long = 0
	Field mclassw:Long = 0
	Field mclassx:Long = 0

	Field mearth:Long = 0
	Field mwater:Long = 0
	Field mwaterterr:Long = 0
	Field mhmc:Long = 0
	Field mhmcterr:Long = 0
	Field mmetal:Long = 0
	Field mammon:Long = 0
	Field mrockyterr:Long = 0
	
	Field mgas1:Long = 0
	Field mgas2:Long = 0
	Field mgas3:Long = 0
	Field mgas4:Long = 0
	Field mgas5:Long = 0
	Field mgasammon:Long = 0
	Field mgaswater:Long = 0
	Field mgashelium:Long = 0
	Field mwatergiant:Long = 0
	
	Field mrocky:Long = 0
	Field micy:Long = 0
	Field mrockicy:Long = 0
	Field mcluster:Long = 0
	
	Field mtotal1:Long = 0
	Field mtotal2:Long = 0
	Field mtotal3:Long = 0
	Field mtotal4:Long = 0
	Field mtotal5:Long = 0
	Field mtotal6:Long = 0
	
	Field pc1:String
	Field pc2:String
	Field pc3:String
	Field pc4:String
	Field pc5:String
	Field pc6:String

	Field cc1:String
	Field cc2:String
	Field cc3:String
	Field cc4:String
	Field cc5:String
	Field cc6:String
		
	Field mtotal:Long = 0
	
	' ----------------------------------------------------------------------------
	' scan and count
	' ----------------------------------------------------------------------------
	Method Scan()
	
		If body Then
		
			' count stars
			Select body.startype
				
				Case "M" classm:+1
				Case "K" classk:+1
				Case "G" classg:+1
				Case "F" classf:+1
				Case "A" classa:+1
				Case "B" classb:+1
				Case "O" classo:+1
				
				Case "N" classn:+1
				Case "H", "SupermassiveBlackHole" classh:+1
				Case "X" classx:+1
				Case "W", "WN", "WNC", "WC", "WO" classw:+1
				Case "CS", "C", "CN", "CJ", "CH", "CHd" classc:+1
				Case "D", "DA", "DAB", "DAO", "DAZ", "DAV", "DB", "DBZ", "DBV", "DO", "DOV", "DQ", "DC", "DCV", "DX" classd:+1
											
				Case "T" classt:+1
				Case "Y" classy:+1
				Case "L" classl:+1
				Case "TTS", "AeBe" classtts:+1
				
			End Select
			
			' count special
			If Instr(body.startype, "SuperGiant") Then supergiant:+1
			If Instr(body.startype, "Giant") Then giant:+1
			If Instr(body.Name, "Belt Cluster") Then cluster:+1
			
			' count planets
			Select body.Class
			
				Case "Earthlike body" earth:+1
				Case "Metal rich body" metal:+1
				Case "Water world" water:+1
				Case "Ammonia world" ammon:+1
				Case "High metal content body" hmc:+1
				
				Case "Rocky body" rocky:+1
				Case "Gas giant with ammonia based life" gasammon:+1
				Case "Gas giant with water based life" gaswater:+1
				Case "Sudarsky class I gas giant" gas1:+1
				Case "Sudarsky class II gas giant" gas2:+1
				Case "Sudarsky class III gas giant" gas3:+1
				Case "Sudarsky class IV gas giant" gas4:+1
				Case "Sudarsky class V gas giant" gas5:+1
				Case "Helium gas giant" gashelium:+1
				Case "Helium rich gas giant" gashelium:+1
				Case "Icy body" icy:+1
				Case "Rocky ice body" rockicy:+1
				Case "Water giant" watergiant:+1
			
			End Select
			
			' count terraformables
			If body.Class = "High metal content body" And body.terraflag Then hmcterr:+1
			If body.Class = "Water world" And body.terraflag Then waterterr:+1
			If body.Class = "Rocky body" And body.terraflag Then rockyterr:+1
			If body.terraflag Then terra:+1
				
		End If
		
	End Method
	
	' ----------------------------------------------------------------------------
	' evaluate results
	' ----------------------------------------------------------------------------
	Method Evaluate()
	
		' substract supergiants from giants because the word giant is already a part of the word supergiant
		giant:-supergiant
		
		' substract terrestrial worlds to get exact results
		water:-waterterr
		hmc:-hmcterr
	
		' counting all together
		total1 = (classy + classt + classl + classtts)
		total2 = (classm + classk + classg + classf + classa + classb + classo)
		total3 = (giant + supergiant + classn + classd + classh + classc + classw + classx)
		total4 = (earth + water + waterterr + hmc + hmcterr + metal + ammon + rockyterr)
		total5 = (rocky + icy + rockicy + cluster)
		total6 = (gas1 + gas2 + gas3 + gas4 + gas5 + gasammon + gaswater + gashelium + watergiant)
		
		total = total1 + total2 + total3 + total4 + total5 + total6

		pc1 = FormatMath(total1 * 100.0 / total, 1)
		pc2 = FormatMath(total2 * 100.0 / total, 1)
		pc3 = FormatMath(total3 * 100.0 / total, 1)
		pc4 = FormatMath(total4 * 100.0 / total, 1)
		pc5 = FormatMath(total5 * 100.0 / total, 1)
		pc6 = FormatMath(total6 * 100.0 / total, 1)

		If pc1 < 0 Then pc1 = FormatMath(0, 1)
		If pc2 < 0 Then pc2 = FormatMath(0, 1)
		If pc3 < 0 Then pc3 = FormatMath(0, 1)
		If pc4 < 0 Then pc4 = FormatMath(0, 1)
		If pc5 < 0 Then pc5 = FormatMath(0, 1)
		If pc6 < 0 Then pc6 = FormatMath(0, 1)

		' multiply counts with rewards
		mclassy = classy * reward_class_y
		mclasst = classt * reward_class_t
		mclassl = classl * reward_class_l
		mclasstts = classtts * reward_class_tts
		
		mclassm = classm * reward_class_m
		mclassk = classk * reward_class_k
		mclassg = classg * reward_class_g
		mclassf = classf * reward_class_f
		mclassa = classa * reward_class_a
		mclassb = classb * reward_class_b
		mclasso = classo * reward_class_o
		
		mgiant = giant * reward_giant
		msupergiant = supergiant * reward_supergiant
		mclassn = classn * reward_class_n
		mclassd = classd * reward_class_d
		mclassh = classh * reward_class_h
		mclassc = classc * reward_class_c
		mclassw = classw * reward_class_w
		mclassx = classx * reward_class_x
	
		mearth = earth * reward_earthlike
		mwater = water * reward_waterworld
		mwaterterr = waterterr * reward_waterworld_terra
		mhmc = hmc * reward_hmc
		mhmcterr = hmcterr * reward_hmc_terra
		mmetal = metal * reward_metalrich
		mammon = ammon * reward_ammonia
		mrockyterr = rockyterr * reward_rocky_terra
		
		mgas1 = gas1 * reward_gas1
		mgas2 = gas2 * reward_gas2
		mgas3 = gas3 * reward_gas3
		mgas4 = gas4 * reward_gas4
		mgas5 = gas5 * reward_gas5
		mgasammon = gasammon * reward_gas_ammonia
		mgaswater = gaswater * reward_gas_water
		mgashelium = gashelium * reward_gas_helium
		mwatergiant = watergiant * reward_gas_watergiant
		
		mrocky = rocky * reward_rocky
		micy = icy * reward_icy
		mrockicy = rockicy * reward_rockyicy
		mcluster = cluster * reward_clusterbelt

		mtotal1 = (mclassy + mclasst + mclassl + mclasstts)
		mtotal2 = (mclassm + mclassk + mclassg + mclassf + mclassa + mclassb + mclasso)
		mtotal3 = (mgiant + msupergiant + mclassn + mclassd + mclassh + mclassc + mclassw + mclassx)
		mtotal4 = (mearth + mwater + mwaterterr + mhmc + mhmcterr + mmetal + mammon)
		mtotal5 = (mrocky + micy + mrockicy + mcluster)
		mtotal6 = (mgas1 + mgas2 + mgas3 + mgas4 + mgas5 + mgasammon + mgaswater + mgashelium + mwatergiant)
		
		mtotal = mtotal1 + mtotal2 + mtotal3 + mtotal4 + mtotal5 + mtotal6
		
		cc1 = FormatMath(mtotal1 * 100.0 / mtotal, 1)
		cc2 = FormatMath(mtotal2 * 100.0 / mtotal, 1)
		cc3 = FormatMath(mtotal3 * 100.0 / mtotal, 1)
		cc4 = FormatMath(mtotal4 * 100.0 / mtotal, 1)
		cc5 = FormatMath(mtotal5 * 100.0 / mtotal, 1)
		cc6 = FormatMath(mtotal6 * 100.0 / mtotal, 1)
		
		If cc1 < 0 Then cc1 = FormatMath(0, 1)
		If cc2 < 0 Then cc2 = FormatMath(0, 1)
		If cc3 < 0 Then cc3 = FormatMath(0, 1)
		If cc4 < 0 Then cc4 = FormatMath(0, 1)
		If cc5 < 0 Then cc5 = FormatMath(0, 1)
		If cc6 < 0 Then cc6 = FormatMath(0, 1)
		
	End Method
	
End Type